<?php

namespace App\Http\Controllers\user;

use App\Http\Controllers\Controller;
use App\Mail\PasswordCreatedEmail;
use App\Mail\UserDomainEnquiryMail;
use App\Models\User;
use Carbon\Carbon;
use Redirect;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\Facades\File;
use mysqli;
use Throwable;
use Illuminate\Support\Str;
use \Mpdf\Mpdf as PDF;
use App\Models\LoginHistory;
use phpseclib3\Net\SFTP;
use Firebase\JWT\JWT;
use Config;
use \Firebase\JWT\Key;
use Log;
use App\Library\EPP;
use App\Models\Zone;
use App\Services\Formatters\BINDFormatter;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Artisan;
use App\Components\MailManager;

class TestController extends Controller
{


    // Test

    public function testHindiDomainAvailability($domainName, $extension = 'भारत')
    {
        try {
            // Decode URL encoded domain name
            $domainName = urldecode($domainName);

            // Full domain with extension
            $fullDomain = $domainName . '.' . $extension;

            // Convert to punycode with error handling
            $punycodeQuery = false;

            if (function_exists('idn_to_ascii')) {
                $punycodeQuery = idn_to_ascii($fullDomain, IDNA_DEFAULT, INTL_IDNA_VARIANT_UTS46);

                // Alternative method if first attempt fails
                if ($punycodeQuery === false) {
                    $punycodeQuery = idn_to_ascii($fullDomain);
                }
            } else {
                throw new \Exception('idn_to_ascii function is not available. Please enable PHP Intl extension.');
            }

            if ($punycodeQuery === false) {
                throw new \Exception('Failed to convert domain to punycode: ' . $fullDomain);
            }

            // Debug information
            $debugInfo = [
                'original_domain' => $fullDomain,
                'punycode_domain' => $punycodeQuery,
                'idn_errors' => [],
            ];

            // Check for IDN conversion errors (if intl extension is available)
            if (function_exists('intl_get_error_code')) {
                $idnError = intl_get_error_code();
                if ($idnError !== 0) {
                    $debugInfo['idn_errors'][] = [
                        'code' => $idnError,
                        'message' => intl_get_error_message()
                    ];
                }
            } else {
                $debugInfo['idn_errors'][] = [
                    'warning' => 'Intl extension not available - cannot check IDN conversion errors'
                ];
            }

            // Initialize EPP
            $epp = new EPP();

            // Check domain availability using punycode
            $response = $epp->checkDomainAvailability($punycodeQuery);

            // Parse EPP response
            $dom = new \DOMDocument();
            $dom->loadXML($response);

            // Extract availability status
            $availability = null;
            $reason = null;

            $cdElements = $dom->getElementsByTagNameNS('urn:ietf:params:xml:ns:domain-1.0', 'cd');
            if ($cdElements->length > 0) {
                $nameElement = $cdElements->item(0)->getElementsByTagName('name')->item(0);
                if ($nameElement) {
                    $availability = $nameElement->getAttribute('avail');

                    // Check for reason (if domain is not available)
                    $reasonElements = $cdElements->item(0)->getElementsByTagName('reason');
                    if ($reasonElements->length > 0) {
                        $reason = $reasonElements->item(0)->textContent;
                    }
                }
            }

            // Check for EPP errors
            $eppErrors = [];
            $resultElements = $dom->getElementsByTagName('result');
            if ($resultElements->length > 0) {
                $resultCode = $resultElements->item(0)->getAttribute('code');
                $msgElements = $resultElements->item(0)->getElementsByTagName('msg');
                if ($msgElements->length > 0) {
                    $eppErrors[] = [
                        'code' => $resultCode,
                        'message' => $msgElements->item(0)->textContent
                    ];
                }
            }

            $result = [
                'success' => true,
                'debug_info' => $debugInfo,
                'epp_response' => [
                    'raw_xml' => $response,
                    'availability' => $availability,
                    'is_available' => $availability === '1',
                    'reason' => $reason,
                    'errors' => $eppErrors
                ],
                'timestamp' => now()->toDateTimeString()
            ];

            // Return JSON response for API or view for web
            if (request()->expectsJson()) {
                return response()->json($result, 200);
            }

            return view('hindi-domain-test', compact('result'));
        } catch (\Exception $e) {
            $error = [
                'success' => false,
                'error' => [
                    'message' => $e->getMessage(),
                    'file' => $e->getFile(),
                    'line' => $e->getLine(),
                    'trace' => $e->getTraceAsString()
                ],
                'debug_info' => $debugInfo ?? [],
                'timestamp' => now()->toDateTimeString()
            ];

            if (request()->expectsJson()) {
                return response()->json($error, 500);
            }

            return view('hindi-domain-test', ['result' => $error]);
        }
    }

    // Enhanced EPP check method with better error handling and intl extension check
    public function checkHindiDomainWithDetails($domainName, $extension)
    {
        $fullDomain = $domainName . '.' . $extension;

        // Step 1: Check if required extensions are available
        if (!function_exists('idn_to_ascii')) {
            return [
                'error' => 'PHP Intl extension is not installed or enabled',
                'domain' => $fullDomain,
                'solution' => 'Enable PHP Intl extension on your server'
            ];
        }

        // Step 2: Validate Hindi characters
        if (!preg_match('/^[\p{Devanagari}\s]+$/u', $domainName)) {
            return [
                'error' => 'Domain contains invalid characters for Hindi',
                'domain' => $fullDomain
            ];
        }

        // Step 3: Convert to punycode with multiple attempts
        $punycodeQuery = idn_to_ascii($fullDomain, IDNA_DEFAULT, INTL_IDNA_VARIANT_UTS46);

        // Fallback to basic conversion if enhanced fails
        if ($punycodeQuery === false) {
            $punycodeQuery = idn_to_ascii($fullDomain);
        }

        // Step 4: Check conversion success
        if ($punycodeQuery === false) {
            $errorMsg = 'Failed to convert to punycode';
            if (function_exists('intl_get_error_message')) {
                $errorMsg .= ': ' . intl_get_error_message();
            }

            return [
                'error' => $errorMsg,
                'domain' => $fullDomain
            ];
        }

        // Step 5: EPP Query
        try {
            $epp = new EPP();
            $eppResponse = $epp->checkDomainAvailability($punycodeQuery);

            return [
                'success' => true,
                'original_domain' => $fullDomain,
                'punycode_domain' => $punycodeQuery,
                'epp_response' => $eppResponse
            ];
        } catch (\Exception $e) {
            return [
                'error' => 'EPP query failed: ' . $e->getMessage(),
                'original_domain' => $fullDomain,
                'punycode_domain' => $punycodeQuery
            ];
        }
    }

    // Blade template for debug view (create resources/views/debug/hindi-domain-test.blade.php)
    // Simple alternative method without intl extension dependency
    public function simpleHindiDomainTest($domainName, $extension = 'भारत')
    {
        try {
            $domainName = urldecode($domainName);
            $fullDomain = $domainName . '.' . $extension;

            // Basic punycode conversion
            $punycodeQuery = idn_to_ascii($fullDomain);

            if ($punycodeQuery === false) {
                return [
                    'success' => false,
                    'error' => 'Failed to convert domain to punycode',
                    'original_domain' => $fullDomain,
                    'suggestions' => [
                        'Check if PHP Intl extension is enabled',
                        'Verify domain contains valid Hindi characters',
                        'Test with a simpler domain name'
                    ]
                ];
            }

            // EPP check
            $epp = new EPP();
            $response = $epp->checkDomainAvailability($punycodeQuery);

            // Simple XML parsing
            $isAvailable = false;
            if (strpos($response, 'avail="1"') !== false) {
                $isAvailable = true;
            }

            return [
                'success' => true,
                'original_domain' => $fullDomain,
                'punycode_domain' => $punycodeQuery,
                'is_available' => $isAvailable,
                'raw_response' => $response,
                'timestamp' => date('Y-m-d H:i:s')
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage(),
                'original_domain' => $fullDomain ?? 'Unknown'
            ];
        }
    }

    // Test

    public function checkEppLogin()
    {
        $host = 'epp.ote.nixiregistry.in';
        $port = 700;

        // $certFile = '/var/www/html/ernet/registry-key/ernet-ote.csr';
        // $keyFile  = '/var/www/html/ernet/registry-key/ernet-ote.key';
        // $caFile   = '/var/www/html/ernet/registry-key/ernet-ote.pem';


        $username = 'ernet_in_a';
        $password = 'c07043de2026ba';

        $certFile = '/var/www/html/ernet/registry-key-test/ernet-ote.csr';
        $keyFile  = '/var/www/html/ernet/registry-key-test/ernet-ote.key';
        $caFile   = '/var/www/html/ernet/registry-key-test/ernet-ote.pem';

        // $username = 'ernet_in_b';
        // $password = 'da73f54517ee33';

        // Validate file paths
        foreach (
            [
                'Certificate' => $certFile,
                'Private Key' => $keyFile,
                'CA Bundle' => $caFile
            ] as $label => $file
        ) {
            if (!file_exists($file)) {
                return response()->json([
                    'status' => 'error',
                    'message' => "$label file not found at $file"
                ]);
            }
        }

        // EPP login XML (without xml declaration per EPP spec)
        $xmlBody = <<<EOT
    <epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
    <command>
        <login>
        <clID>{$username}</clID>
        <pw>{$password}</pw>
        <options>
            <version>1.0</version>
            <lang>en</lang>
        </options>
        <svcs>
            <objURI>urn:ietf:params:xml:ns:domain-1.0</objURI>
            <objURI>urn:ietf:params:xml:ns:contact-1.0</objURI>
            <objURI>urn:ietf:params:xml:ns:host-1.0</objURI>
        </svcs>
        </login>
        <clTRID>ABC-12345</clTRID>
    </command>
    </epp>
    EOT;

        // Add 4-byte length prefix as per EPP framing (length includes the 4 length bytes)
        $length = strlen($xmlBody) + 4;
        $prefix = pack('N', $length); // big-endian 32-bit unsigned int
        $xml = $prefix . $xmlBody;

        $context = stream_context_create([
            'ssl' => [
                'local_cert'        => $certFile,
                'local_pk'          => $keyFile,
                'cafile'            => $caFile,
                'verify_peer'       => true,
                'verify_peer_name'  => true,
                'allow_self_signed' => false,
            ]
        ]);

        $fp = @stream_socket_client(
            "tls://{$host}:{$port}",
            $errno,
            $errstr,
            30,
            STREAM_CLIENT_CONNECT,
            $context
        );

        if (!$fp) {
            return response()->json([
                'status' => 'error',
                'message' => "Failed to connect: $errstr ($errno)"
            ]);
        }

        stream_set_timeout($fp, 30);

        // Read the EPP greeting (length-prefixed)
        $greetingLengthRaw = fread($fp, 4);
        if (strlen($greetingLengthRaw) < 4) {
            fclose($fp);
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to read greeting length from server'
            ]);
        }
        $greetingLength = unpack('N', $greetingLengthRaw)[1];
        $greetingBody = fread($fp, $greetingLength - 4);

        // Send the login XML with prefix
        fwrite($fp, $xml);

        // Read the login response (length-prefixed)
        $responseLengthRaw = fread($fp, 4);
        if (strlen($responseLengthRaw) < 4) {
            fclose($fp);
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to read login response length from server'
            ]);
        }
        $responseLength = unpack('N', $responseLengthRaw)[1];
        $responseBody = fread($fp, $responseLength - 4);

        fclose($fp);

        return response()->json([
            'status' => 'success',
            'greeting' => trim($greetingBody),
            'login_response' => trim($responseBody)
        ]);
    }
    
}
