<?php

namespace App\Library;
use Config;
use DB,Log;
use App\Models\User;
use Exception;
use RuntimeException;
class EPP
{

    private $host;
    private $port;
    private $username;
    private $password;
    private $timeout;
    private $loggedIn;
    
    private $connected;  
    
    private $params;
    private $socket;

    
    public function __construct()
    {
        $this->timeout = 300;
        $this->socket = null;
        $this->connected = false;
        $this->loggedIn=false;
        $this->username=Config::get('custom-epp.username');
        $this->password=Config::get('custom-epp.password');
        $this->connect();
        $this->login($this->username,$this->password);
    }

    public function connect($passphrase=null,$verify_peer=false,$verify_peer_name=false,$verify_host=false,$allow_self_signed=true)
    {
        
        $host = Config::get('custom-epp.host');
        $port = Config::get('custom-epp.port');
        $username=Config::get('custom-epp.username');
        $password=Config::get('custom-epp.password');
        $certificate=Config::get('custom-epp.certificate_file');
        $privatekey=Config::get('custom-epp.private_key');

        $timeout=10000;
        $tls_version = '1.3';
        $opts = [
            'ssl' => [
                'verify_peer' => $verify_peer,
                'verify_peer_name' => $verify_peer_name,
                'verify_host' => $verify_host,
                'local_cert' => $certificate,
                'local_pk' => $privatekey,
                'passphrase' => $passphrase,
                'allow_self_signed' => $allow_self_signed
            ]
        ];

        if($this->isConnected()==false){

        $target = "tls://{$host}:{$port}";
        $context = stream_context_create($opts);

        $this->socket = stream_socket_client($target, $errno, $errmsg, $timeout, STREAM_CLIENT_CONNECT, $context);

        if (!is_resource($this->socket)) {
            return "Connecting to " . $target . ". <p>The error message was '" . $errmsg . "' (code " . $errno . ")";
        }

        $res=$this->read("connect");

        $this->connected = true;

        return $this->socket;

        }

        return true;

    }

    public function setParams($params)
    {
        $this->params = $params;
    }

    public function read($action)
    {
        $hdr = stream_get_contents($this->socket, 4);
        if ($hdr === false) {
            return 'Connection appears to have closed.';
        }
        if (strlen($hdr) < 4) {
            return 'Failed to read header from the connection.';
        }
        $unpacked = unpack('N', $hdr);
        $xml = fread($this->socket, ($unpacked[1] - 4));
        $xml = preg_replace('/></', ">\n<", $xml);
        return $xml;

    }

    public function isConnected()
    {
        return $this->connected;
    }

    public function isLoggedIn()
    {
        return $this->loggedIn;
    }

    public function login($username,$password,$newPwd=null){
        
        $from = $to = [];
        $from[] = '/{{ clID }}/';
        $to[] = '<clID>' . htmlspecialchars($username) . '</clID>';

        $from[] = '/{{ pw }}/';
        $to[] = '<pw>' . $this->htmldecoder($password) . '</pw>';

        $from[] = '/{{ newPW }}/';
        $to[] = $newPwd === null ? '' : '<newPW>' . $this->htmldecoder($newPwd) . '</newPW>';

        $from[] = '/{{ clTRID }}/';
        $clTRID = str_replace('.', '', round(microtime(1), 3));
        $to[] = htmlspecialchars('er' . '-login-' . $clTRID);

        $xml = preg_replace($from, $to, "
        <epp xmlns='urn:ietf:params:xml:ns:epp-1.0'>
                    <command>
        <login>
            {{ clID }}
            {{ pw }}
            {{ newPW }}
            <options>
                <version>1.0</version>
                <lang>en</lang>
            </options>
            <svcs>
                <objURI>urn:ietf:params:xml:ns:domain-1.0</objURI>
                <objURI>urn:ietf:params:xml:ns:contact-1.0</objURI>
                <objURI>urn:ietf:params:xml:ns:host-1.0</objURI>
                <svcExtension>
                    <extURI>urn:ietf:params:xml:ns:idn-1.0</extURI>
                    <extURI>urn:ietf:params:xml:ns:secDNS-1.1</extURI>
                </svcExtension>
            </svcs>
        </login>
        <clTRID>{{ clTRID }}</clTRID>
    </command>
    </epp>");

        $res = $this->runCommand($xml, __FUNCTION__);
        $this->loggedIn=true;
        return $res;

    }

    public function htmldecoder($str){
        return htmlspecialchars(htmlspecialchars_decode($str));
    }

    private function isPunycodeDomain($domain)
    {
        return strpos($domain, 'xn--') !== false;
    }

   private function generateTransactionId()
    {
        return 'DOM-' . date('YmdHis') . '-' . rand(1000, 9999);
    }

    private function runCommand($xmlcommand,$action='Null',$fullXml=false){

            if (fwrite($this->socket, pack('N', (strlen($xmlcommand) + 4)) . $xmlcommand) === false) {
                return 'Error writing to the connection.';
            }

            $response = $this->read($action);
            return $response;

    }

    public function disconnect()
    {
        if ($this->socket) {
            fclose($this->socket);
            $this->socket = null;
            $this->connected = false;
        };
        return true;
    }

    public function checkDomainAvailability($domainName)
    {
        // Validate domain name before sending to EPP
        if (empty($domainName) || strlen($domainName) > 253) {
            throw new \Exception('Invalid domain name length');
        }
        
        // Escape domain name for XML
        $escapedDomain = htmlspecialchars($domainName, ENT_XML1, 'UTF-8');
        
        $xml = "<epp xmlns='urn:ietf:params:xml:ns:epp-1.0'>
                    <command>
                        <check>
                            <domain:check xmlns:domain='urn:ietf:params:xml:ns:domain-1.0'>
                                <domain:name>{$escapedDomain}</domain:name>
                            </domain:check>
                        </check>
                        <clTRID>" . $this->generateTransactionId() . "</clTRID>
                    </command>
                </epp>";

        try {
            $response = $this->runCommand($xml, __FUNCTION__, true);
            
           
            if (empty($response)) {
                throw new \Exception('Empty response from EPP server');
            }
            
            
            if ($this->isPunycodeDomain($domainName)) {
                \Log::info('EPP Hindi Domain Check', [
                    'domain' => $domainName,
                    'response_length' => strlen($response)
                ]);
            }
            
            return $response;
            
        } catch (\Exception $e) {
            
            \Log::error('EPP Command Error', [
                'domain' => $domainName,
                'function' => __FUNCTION__,
                'error' => $e->getMessage()
            ]);
            
            throw $e;
        }
    }

    // New EPP Sets for the Data Policy Fixing

    public function create_domain_v2($domainName, $periodYr, $email, $password, $unique)
    {
        $user = User::find($unique);
        $usercontact = DB::table('user_contact_details')->where('user_id', $user->id)->first();

        
        if (empty($user->institute) || empty($usercontact->address) || 
            empty($usercontact->city) || empty($usercontact->state) || 
            empty($usercontact->pincode)) {
            throw new Exception("Required contact information is missing");
        }

        $contactsXml = "";
        $contactTypes = ['registrant', 'admin', 'tech', 'billing'];
        
        foreach ($contactTypes as $contactType) {
            $contactId = $this->create_contact_v2($user, $usercontact, $contactType, $password);
            if ($contactType == 'registrant') {
                $contactsXml .= '<domain:registrant>' . $contactId . '</domain:registrant>';
            } else {
                $contactsXml .= '<domain:contact type="' . $contactType . '">' . $contactId . '</domain:contact>';
            }
        }

        return $this->send_domain_create_v2($domainName, $periodYr, $contactsXml, $password);
    }
    protected function create_contact_v2($user, $usercontact, $contactType, $password)
    {
        $prefixMap = [
            'registrant' => 'R',
            'admin' => 'A',
            'tech' => 'T',
            'billing' => 'B'
        ];
        
        $contactId = substr($user->domainName, 0, 3) . $prefixMap[$contactType] . rand(1000000, 9999999);

        
        $details = $this->get_contact_details_v2($user, $usercontact, $contactType);
        
        $params = [
            'name' => $this->strict_sanitize_ascii($details['name']),
            'org' => $this->strict_sanitize_ascii($user->institute),
            'address' => $this->strict_sanitize_ascii($details['address']),
            'city' => $this->strict_sanitize_ascii($details['city']),
            'state' => $this->strict_sanitize_ascii($details['state']),
            'postcode' => $this->validate_postcode($details['postcode']),
            'country' => 'IN',
            'mobile' => $this->format_phone($details['mobile']),
            'email' => $this->validate_email($details['email']),
            'password' => $password
        ];

        // Create the contact
        $response = $this->contactTypeCreate($contactId, $params);
        $xml = simplexml_load_string($response);
        
        // Check for errors
        if ((string)$xml->response->result->attributes()->code != '1000') {
            throw new Exception("Failed to create $contactType contact: " . (string)$xml->response->result->msg);
        }

        return $contactId;
    }

    protected function get_contact_details_v2($user, $usercontact, $contactType)
    {
        $details = [
            'name' => $user->name,
            'address' => $usercontact->address,
            'city' => $usercontact->city,
            'state' => $usercontact->state,
            'postcode' => $usercontact->pincode,
            'mobile' => $usercontact->mobile,
            'email' => $user->email
        ];

        // $details = [
        //                 'name' => $user->name ?: 'Registrant Name',
        //                 'address' => $usercontact->address,
        //                 'city' => $usercontact->city,
        //                 'state' => $usercontact->state,
        //                 'postcode' => $usercontact->pincode,
        //                 'mobile' => $usercontact->mobile ?: '0000000000',
        //                 'email' => $user->email ?: 'registrant@example.com'
        //             ];

        if ($contactType == 'admin') {
            $value = json_decode($usercontact->administrative_detail, true) ?? [];
            $details = array_merge($details, [
                'name' => $value['name'] ?? '',
                'email' => $value['email'] ?? '',
                'mobile' => $value['mobile'] ?? ''
            ]);
        } elseif ($contactType == 'tech') {
            $value = json_decode($usercontact->technical_detail, true) ?? [];
            $details = array_merge($details, [
                'name' => $value['name'] ?? '',
                'email' => $value['email'] ?? '',
                'mobile' => $value['mobile'] ?? ''
            ]);
        } elseif ($contactType == 'billing') {
            $value = json_decode($usercontact->billing_contact_detail, true) ?? [];
            $details = array_merge($details, [
                'name' => $value['name'] ?? '',
                'email' => $value['email'] ?? '',
                'mobile' => $value['mobile'] ?? ''
            ]);
        }

        
        foreach (['name', 'email', 'mobile'] as $field) {
            if (empty($details[$field])) {
                throw new Exception("$contactType contact $field is required");
            }
        }

        return $details;
    }
    protected function strict_sanitize_ascii($string, $maxLength = null)
    {
        // First remove non-ASCII characters
        $string = preg_replace('/[^\x20-\x7E]/', '', $string);
        
        // Then escape XML special characters
        $string = htmlspecialchars($string, ENT_XML1 | ENT_QUOTES, 'UTF-8', false);
        
        // Remove multiple spaces and trim
        $string = preg_replace('/\s+/', ' ', trim($string));
        
        if ($maxLength) {
            $string = substr($string, 0, $maxLength);
        }
        
        return $string;
    }
    protected function format_phone($phone)
    {
        $phone = preg_replace('/[^0-9]/', '', $phone);
        if (strlen($phone) != 10) {
            throw new Exception("Phone number must be 10 digits");
        }
        return '+91.' . $phone;
    }
    protected function validate_email($email)
    {
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            throw new Exception("Invalid email address");
        }
        return $email;
    }
    protected function validate_postcode($postcode)
    {
        $postcode = preg_replace('/[^0-9]/', '', $postcode);
        if (strlen($postcode) != 6) {
            throw new Exception("Postal code must be 6 digits");
        }
        return $postcode;
    }
    protected function send_domain_create_v2($domainName, $periodYr, $contactsXml, $authInfo)
    {
        $clTRID = 'ernet-domain-create-' . round(microtime(true) * 1000);
        
        $xml = <<<XML
    <epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
    <command>
        <create>
        <domain:create xmlns:domain="urn:ietf:params:xml:ns:domain-1.0">
            <domain:name>{$domainName}</domain:name>
            <domain:period unit="y">{$periodYr}</domain:period>
            {$contactsXml}
            <domain:authInfo>
            <domain:pw>{$authInfo}</domain:pw>
            </domain:authInfo>
        </domain:create>
        </create>
        <clTRID>{$clTRID}</clTRID>
    </command>
    </epp>
    XML;

        return $this->runCommand($xml, 'create_domain_v2', true);
    }



    // New EPP

    public function checkDomainAvailabilityOld($domainName)
    {

        $xml = "<epp xmlns='urn:ietf:params:xml:ns:epp-1.0'>
                        <command>
                            <check>
                                <domain:check xmlns:domain='urn:ietf:params:xml:ns:domain-1.0'>
                                    <domain:name>$domainName</domain:name>
                                </domain:check>
                            </check>
                            <clTRID>ABC-12345</clTRID>
                        </command>
                    </epp>";


        $response=$this->runCommand($xml, __FUNCTION__,true);
        return  $response; 

    }
    
    public function domainInfo($domainName){
        
        $xml='<epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
        <command>
        <info>
        <info xmlns="urn:ietf:params:xml:ns:domain-1.0">
        <name>'.$domainName.'</name>
        </info>
        </info>
        </command>
        </epp>';

        $response=$this->runCommand($xml, __FUNCTION__,true);
        return  $response; 

    }


    public function register_domain_old($domainName,$password,$contacts){

        $contactsXml = "";
        if (!empty($contacts)) {
            foreach ($contacts as $contactType => $contactID) {

                if ($contactType == 'registrant') {
                    $contactsXml .= '<domain:registrant>' . $contactID . '</domain:registrant>';
                } else {
                    $contactsXml .= '<domain:contact type="' . $contactType . '">' . $contactID . '</domain:contact>' . "\n";
                }
            }
        }

        $from[] = '/{{ contacts }}/';
        $to[] = $contactsXml;

        $xml='<epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
        <command>
        <create>
        <create xmlns="urn:ietf:params:xml:ns:domain-1.0">
        <name>'.$domainName.'</name>
        <registrant>REQUIREDCONTACT</registrant>
        <contact type="tech">TECHCONTACT</contact>
        <contact type="biling">TECHCONTACT</contact>
        <contact type="admin">TECHCONTACT</contact>
        <authInfo>
        <pw>ERNET-S01</pw>
        </authInfo>
        </create>
        </create>
        <clTRID>ABC-12345</clTRID>
        </command>
        </epp>';

        $response=$this->runCommand($xml, __FUNCTION__,true);
        return  $response; 

    }


    public function deletedomain($domainName){
        
        $cltRID=str_replace('.', '', round(microtime(1), 3));

        $xml='<epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
        <command>
        <delete>
        <delete xmlns="urn:ietf:params:xml:ns:domain-1.0">
        <name>'.$domainName.'</name>
        </delete>
        </delete>
        <clTRID>'.$cltRID.'</clTRID>
        </command>
        </epp>';

        $response=$this->runCommand($xml, __FUNCTION__,true);
        return  $response; 

    }


    public function create_domain($domainName, $periodYr,$email,$password,$unique){

    $user = User::find($unique);

    $usercontact=DB::table('user_contact_details')->select('*')->where('user_id',$user->id)->first();

    $contactsXml="";

    $contents = [
        'registrant',
        'admin',
        'tech',
        'billing',
    ];
    
    foreach ($contents as $contactType) {

        $org=$user->institute;
        $name='';
        $address=$usercontact->address;
        $city=$usercontact->city;
        $state=$usercontact->state;
        $postcode=$usercontact->pincode;
        $email='';
        $mobileno='';

        if($contactType=='registrant'){
            $uniquecid=substr($user->domainName,0,3).'R'.rand(1999045,9934777);

            $name=$user->name;
            $email=$user->email;
            $mobileno=$usercontact->mobile;
          
        }else if($contactType=='admin'){
            $uniquecid=substr($user->domainName,0,3).'A'.rand(1999045,9934777);

            $value=json_decode($usercontact->administrative_detail,true);

            $name=$value['name']??'';
            $email=$value['email']??'';
            $mobileno=$value['mobile']??'';

        }else if($contactType=='tech'){
            $uniquecid=substr($user->domainName,0,3).'T'.rand(1999045,9934777);

            $value=json_decode($usercontact->technical_detail,true);

            $name=$value['name']??'';
            $email=$value['email']??'';
            $mobileno=$value['mobile']??'';

        }else if($contactType=='billing'){
            $uniquecid=substr($user->domainName,0,3).'B'.rand(1999045,9934777);

            $value=json_decode($usercontact->billing_contact_detail,true);

            $name=$value['name']??'';
            $email=$value['email']??'';
            $mobileno=$value['mobile']??'';

        }

        $modifiedText = str_replace("\n", "", $address);

        $params=[
            'name'=>$this->sanitize_ascii($name),
            'org'=>$this->removeSpecialCharacters($org),
            'address'=>$this->removeSpecialCharacters($address),
            'city'=>$this->removeSpecialCharacters($city),
            'state'=>$this->removeSpecialCharacters($state),
            'postcode'=>$postcode,
            'country'=>'IN',
            'mobile'=>'+91.'.$mobileno,
            'email'=>$email,
            'password'=>$password
        ];    
        

        $test = $this->contactTypeCreate($uniquecid,$params);            

        if ($contactType == 'registrant') {
            $contactsXml .= '<domain:registrant>' . $uniquecid . '</domain:registrant>';
        } else {
            $contactsXml .= '<domain:contact type="' . $contactType . '">' . $uniquecid . '</domain:contact>' . "\n";
        }
      
    }

        $authInfo=$password;
        $from = $to = [];

        $from[] = '/{{ name }}/';
        $to[] = htmlspecialchars($domainName);

        $from[] = '/{{ period }}/';
        $to[] = is_numeric($periodYr) ? $periodYr : 1;

        $from[] = '/{{ contacts }}/';
        $to[] = $contactsXml;
    
        $from[] = '/{{ authInfoPw }}/';
        $to[] = htmlspecialchars($authInfo);

        $from[] = '/{{ clTRID }}/';
        $clTRID = str_replace('.', '', round(microtime(1), 3));
        $to[] = htmlspecialchars("ernet" . '-domain-create-' . $clTRID);

        $from[] = "/<\w+:\w+>\s*<\/\w+:\w+>\s+/ims";
        $to[] = '';

        $xml = preg_replace($from, $to, '<epp xmlns="urn:ietf:params:xml:ns:epp-1.0"><command>
    <create>
      <domain:create
       xmlns:domain="urn:ietf:params:xml:ns:domain-1.0">
        <domain:name>{{ name }}</domain:name>
        <domain:period unit="y">{{ period }}</domain:period>
        {{ contacts }}
        <domain:authInfo>
          <domain:pw>{{ authInfoPw }}</domain:pw>
        </domain:authInfo>
      </domain:create>
    </create>
    <clTRID>{{ clTRID }}</clTRID>
  </command>
  </epp>');

  $response=$this->runCommand($xml, __FUNCTION__,true);
  return  $response;   
//   dd($response);

    }

    private function sanitize_ascii($input) {
        return preg_replace('/[^\x20-\x7E]/', '', $input);
    }

    private function removeSpecialCharacters($string) {
        $cleanedString = preg_replace('/[^a-zA-Z0-9-, \s]/', '', $string);

        // $limitedString = substr($cleanedString, 0, 95);

        return $cleanedString;

    }

    public function domainUpdateContact($domainName, $contactType, $newContactId = null, $oldContactId = null)
    {
       
        $chgXML = $addXML = $remXML = "";
        if ($contactType === 'registrant') {
            $chgXML = '<domain:chg><domain:registrant>' . htmlspecialchars($newContactId) . '</domain:registrant></domain:chg>';
        } else {
            $addXML = $newContactId === null ? '' : '<domain:add><domain:contact type="' . htmlspecialchars($contactType) . '">' . htmlspecialchars($newContactId) . '</domain:contact></domain:add>';
            $remXML = $oldContactId === null ? '' : '<domain:rem><domain:contact type="' . htmlspecialchars($contactType) . '">' . htmlspecialchars($oldContactId) . '</domain:contact></domain:rem>';
        }

        $from = $to = [];
        $from[] = '/{{ name }}/';
        $to[] = htmlspecialchars($domainName);

        $from[] = '/{{ add }}/';
        $to[] = $addXML;
        $from[] = '/{{ rem }}/';
        $to[] = $remXML;
        $from[] = '/{{ chg }}/';
        $to[] = $chgXML;

        $from[] = '/{{ clTRID }}/';
        $clTRID = str_replace('.', '', round(microtime(1), 3));
        $to[] = htmlspecialchars('domain-updateContact-' . $clTRID);

        $xml = preg_replace($from, $to, '<epp xmlns="urn:ietf:params:xml:ns:epp-1.0"><command>
        <update>
          <domain:update
           xmlns:domain="urn:ietf:params:xml:ns:domain-1.0">
            <domain:name>{{ name }}</domain:name>
            {{ add }}
            {{ rem }}
            {{ chg }}
          </domain:update>
        </update>
        <clTRID>{{ clTRID }}</clTRID>
        </command>
        </epp>');

        $response=$this->runCommand($xml, __FUNCTION__,true);

        return $response;

    }

    public function renewdomain($domainName,$expirydate,$period){

        $xml='<epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
        <command>
        <renew>
        <renew xmlns="urn:ietf:params:xml:ns:domain-1.0">
        <name>'.$domainName.'</name>
        <curExpDate>'.$expirydate.'</curExpDate>
        <period unit="y">'.$period.'</period>
        </renew>
        </renew>
        </command>
        </epp>';

        $response=$this->runCommand($xml, __FUNCTION__,true);
        return  $response;   

    }

    public function domainTransfer($opType,$domainName,$period,$password){

        $from = $to = [];
        $authInfo=$password;

        $from[] = '/{{ name }}/';
        $to[] = htmlspecialchars($domainName);

        $from[] = '/{{ type }}/';
        $to[] = htmlspecialchars($opType);

        if ($opType === "request") {

            $from[] = '/{{ periodYr }}/';
            $periodYr = $period;
            $to[] = '<domain:period unit="y">' . $periodYr . '</domain:period>';

            $from[] = '/{{ authInfo }}/';
            $to[] = !empty($authInfo) ? '<domain:authInfo><domain:pw>' . $this->htmldecoder($authInfo) . '</domain:pw></domain:authInfo>' : "";

        } else {
            $from[] = '/{{ periodYr }}/';
            $to[] = "";
            $from[] = '/{{ authInfo }}/';
            $to[] = "";
        }

        $from[] = '/{{ clTRID }}/';
        $clTRID = str_replace('.', '', round(microtime(1), 3));
        $to[] = htmlspecialchars('transfer' . '-domain-transfer-' . $opType . '-' . $clTRID);

        $xml=preg_replace($from, $to, '<epp xmlns="urn:ietf:params:xml:ns:epp-1.0"><command>
        <transfer op="{{ type }}">
          <domain:transfer
           xmlns:domain="urn:ietf:params:xml:ns:domain-1.0">
            <domain:name>{{ name }}</domain:name>
            {{ periodYr }}
            {{ authInfo }}
          </domain:transfer>
        </transfer>
        <clTRID>{{ clTRID }}</clTRID>
      </command></epp>');

       $response=$this->runCommand($xml, __FUNCTION__,true);
       return  $response;  

    }

    public function getParams($key)
    {
        if (isset($this->params[$key])) {
            return $this->params[$key];
        } else {
        }
    }


public function contactCreate($contactIdentifier,$params,$email,$password)
{  
    $this->setParams($params);
    $from = $to = [];
    $from[] = '/{{ type }}/';
    $to[] = (isset($params['postalInfo_type']) ? htmlspecialchars($params['postalInfo_type']) : "int");
    $from[] = '/{{ id }}/';
    $to[] = strtoupper($contactIdentifier);
    $from[] = '/{{ name }}/';
    $to[] = htmlspecialchars($this->getParams('firstname') . ' ' . ($params['lastname'] ?? ''));
    $from[] = '/{{ org }}/';
    $to[] = htmlspecialchars($this->getParams('companyname'));
    $from[] = '/{{ street1 }}/';
    $to[] = htmlspecialchars($this->getParams('address1'));
    $from[] = '/{{ street2 }}/';
    $to[] = (isset($params['address2']) ? htmlspecialchars($this->getParams('address2')) : '');
    $from[] = '/{{ street3 }}/';
    $street3 = (isset($params['address3']) ? htmlspecialchars($this->getParams('address3')) : '');
    $to[] = htmlspecialchars($street3);
    $from[] = '/{{ city }}/';
    $to[] = htmlspecialchars($this->getParams('city'));
    $from[] = '/{{ state }}/';
    $to[] = htmlspecialchars($this->getParams('state'));
    $from[] = '/{{ postcode }}/';
    $to[] = htmlspecialchars($this->getParams('postcode'));
    $from[] = '/{{ country }}/';
    $to[] = htmlspecialchars($this->getParams('country'));
    $from[] = '/{{ phonenumber }}/';
    $to[] = htmlspecialchars($this->getParams('fullphonenumber'));
    $from[] = '/{{ fax }}/';
    $to[] = (isset($params['fax']) ? htmlspecialchars($this->getParams('fax')) : "");
    $from[] = '/{{ email }}/';
    $to[] = htmlspecialchars($this->getParams('email'));
    $from[] = '/{{ authInfo }}/';
    $to[] = htmlspecialchars($password);
    $from[] = '/{{ clTRID }}/';
    $clTRID = str_replace('.', '', round(microtime(1), 3));
    $to[] = htmlspecialchars('er' . '-contact-create-' . $clTRID);
    $from[] = "/<\w+:\w+>\s*<\/\w+:\w+>\s+/ims";
    $to[] = '';

    $xml = preg_replace($from, $to, '<epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
    <command>
    <create>
    <create xmlns="urn:ietf:params:xml:ns:contact-1.0">
    <id>{{ id }}</id>
    <postalInfo type="int">
    <name>{{name}}</name>
    <org>{{org}}</org>
    <addr>
    <street>{{address1}}</street>
    <street>{{address2}}</street>
    <street>{{address3}}</street>
    <city>{{city}}</city>
    <sp>{{state}}</sp>
    <pc>{{postcode}}</pc>
    <cc>IN</cc>
    </addr>
    </postalInfo>
    <email>'.$email.'</email>
    <authInfo>
    <pw>'.$password.'</pw>
    </authInfo>
    </create>
    </create>
    <clTRID>{{clTRID}}</clTRID></command></epp>');

    $r = $this->runCommand($xml, __FUNCTION__);

    return $r;

}


public function contactTypeCreate($contactIdentifier,$params)
{  
   
    $from = $to = [];
    $from[] = '/{{ clTRID }}/';
    $clTRID = str_replace('.', '', round(microtime(1), 3));
    $to[] = htmlspecialchars('er' . '-contact-create-' . $clTRID);
    $from[] = "/<\w+:\w+>\s*<\/\w+:\w+>\s+/ims";
    $to[] = '';
    
    $xml = preg_replace($from, $to, '<epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
    <command>
    <create>
    <create xmlns="urn:ietf:params:xml:ns:contact-1.0">
    <id>'.$contactIdentifier.'</id>
    <postalInfo type="int">
    <name>'.$params['name'].'</name>
    <org>'.$params['org'].'</org>
    <addr>
    <street>'.$params['address'].'</street>
    <city>'.$params['city'].'</city>
    <sp>'.$params['state'].'</sp>
    <pc>'.$params['postcode'].'</pc>
    <cc>IN</cc>
    </addr>
    </postalInfo>
    <voice>'.$params['mobile'].'</voice>
    <email>'.$params['email'].'</email>
    <authInfo>
    <pw>'.$params['password'].'</pw>
    </authInfo>
    </create>
    </create>
    <clTRID>{{clTRID}}</clTRID></command></epp>');
    

    $r = $this->runCommand($xml, __FUNCTION__);

    // \Log::debug("EPP Contact Create XML:\n" . $xml);

    return $r;

}


public function domainRestoreRequest($domainName){
    
    $cltRID=str_replace('.', '', round(microtime(1), 3));

    $xml='<epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
    <command>
    <update>
    <update xmlns="urn:ietf:params:xml:ns:domain-1.0">
    <name>'.$domainName.'</name>
    </update>
    </update>
    <extension>
    <update xmlns="urn:ietf:params:xml:ns:rgp-1.0">
    <restore op="request" />
    </update>
    </extension>
    <clTRID>'.$cltRID.'</clTRID>
    </command>
   </epp>
   ';

    $response=$this->runCommand($xml, __FUNCTION__,true);
    return  $response; 

}


public function domainRestoreReport($domainName,$deldate,$currentdate){
    
    $cltRID=str_replace('.', '', round(microtime(1), 3));

    $xml='<epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
    <command>
    <update>
    <update xmlns="urn:ietf:params:xml:ns:domain-1.0">
    <name>'.$domainName.'</name>
    </update>
    </update>
    <extension>
    <update xmlns="urn:ietf:params:xml:ns:rgp-1.0">
    <restore op="report">
    <report>
    <preData>
     Payment Received from User
    </preData>
    <postData>
    Payment Received from User
    </postData>
    <delTime>'.$deldate.'</delTime>
    <resTime>'.$currentdate.'</resTime>
    <resReason>Payment Received from User</resReason>
    <statement>Payment Received from User</statement>
    <statement>Payment Received from User</statement>
    </report>
    </restore>
    </update>
    </extension>
    <clTRID>'.$cltRID.'</clTRID>
    </command>
   </epp>
   ';

    $response=$this->runCommand($xml, __FUNCTION__,true);
    return  $response; 

}

public function contactInfo($contactID){

    $cltRID=str_replace('.', '', round(microtime(1), 3));

    $xml='<epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
    <command>
    <info>
    <info xmlns="urn:ietf:params:xml:ns:contact-1.0">
      <id>'.$contactID.'</id>
    </info>
    </info>
    <clTRID>'.$cltRID.'</clTRID>
    </command>
   </epp>
   ';

    $response=$this->runCommand($xml, __FUNCTION__,true);
    return  $response; 

}

public function contactDelete($contactID){

$cltRID=str_replace('.', '', round(microtime(1), 3));

$xml='<epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
<command>
<delete>
 <delete xmlns="urn:ietf:params:xml:ns:contact-1.0">
 <id>'.$contactID.'</id>
 </delete>
</delete>
<clTRID>'.$cltRID.'</clTRID>
</command>
</epp>
';

$response=$this->runCommand($xml, __FUNCTION__,true);
return  $response; 

}

public function contactUpdate($contactID,$params){

    $from = $to = [];
    $from[] = '/{{ clTRID }}/';
    $clTRID = str_replace('.', '', round(microtime(1), 3));
    $to[] = htmlspecialchars('er' . '-contact-update-' . $clTRID);
    $from[] = "/<\w+:\w+>\s*<\/\w+:\w+>\s+/ims";
    $to[] = '';
    
    $xml = preg_replace($from, $to, '<epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
    <command>
    <update>
    <update xmlns="urn:ietf:params:xml:ns:contact-1.0">
    <id>'.$contactID.'</id>
    <chg>
    <postalInfo type="int">
    <name>'.$params['name'].'</name>
    <org>'.$params['org'].'</org>
    <addr>
    <street>'.$params['address'].'</street>
    <city>'.$params['city'].'</city>
    <sp>'.$params['state'].'</sp>
    <pc>'.$params['postcode'].'</pc>
    <cc>IN</cc>
    </addr>
    </postalInfo>
    <voice>'.$params['mobile'].'</voice>
    <email>'.$params['email'].'</email>
    </chg>
    </update>
    </update>
    <clTRID>{{clTRID}}</clTRID></command></epp>');
    

    $r = $this->runCommand($xml, __FUNCTION__);

    return $r;

}

private function createXMLByKey($data,$keyname){

    $return='';

    if(empty($data)){

    }else if(is_array($data) & count($data)>0){

        foreach ($data as $v) {
            if (!empty($v)) {
                $fv = htmlspecialchars($v);
                $return .= $keyname === null ? $fv : '<' . $keyname . '>' . $fv . '</' . $keyname . '>';
            }
        }

    }else{
        $return.='';
    }

    return $return; 

}

public function hostCreate($name,$ipv4=null,$ipv6=null){

$cltRID=str_replace('.', '', round(microtime(1), 3));
if(($ipv6==':::::::' || $ipv6=':::::' || $ipv6==null ) && $ipv4!=null){

$xml='<epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
<command>
<create>
  <host:create
   xmlns:host="urn:ietf:params:xml:ns:host-1.0">
    <host:name>'.$name.'</host:name>
    <host:addr ip="v4">'.$ipv4.'</host:addr>
  </host:create>
</create>
<clTRID>'.$cltRID.'</clTRID>
</command>
</epp>
';

}else if($ipv4==null && ($ipv6==':::::' || $ipv6==null || $ipv6=':::::::'))
{
    
$xml='<epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
<command>
<create>
  <host:create
   xmlns:host="urn:ietf:params:xml:ns:host-1.0">
    <host:name>'.$name.'</host:name>
  </host:create>
</create>
<clTRID>'.$cltRID.'</clTRID>
</command>
</epp>
';
    
}else{

$xml='<epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
<command>
<create>
  <host:create
   xmlns:host="urn:ietf:params:xml:ns:host-1.0">
    <host:name>'.$name.'</host:name>
    <host:addr ip="v4">'.$ipv4.'</host:addr>
    <host:addr ip="v6">'.$ipv6.'</host:addr>
  </host:create>
</create>
<clTRID>'.$cltRID.'</clTRID>
</command>
</epp>
';

}

$response=$this->runCommand($xml, __FUNCTION__,true);


return $response;

}


public function hostUpdate($name,$ipv4=null){

    $res=$this->hostInfo($name);

    $output=xmlConverter($res);

    $r=$output['epp']['response'];

    $ipold=$r['resData']['host:infData']['host:addr']??'';

    $validPatterns = '(\.ac\.in|\.edu\.in|\.res\.in)$';

    if (preg_match('/' . $validPatterns . '/', $name)) {

        if(is_string($ipold)){

            $from = $to = [];
            $from[] = '/{{ name }}/';
            $to[] = htmlspecialchars($name);
        
            $from[] = '/{{ addIp }}/';
            $to[] ='<host:add>
            <host:addr ip="v4">'.$ipv4.'</host:addr>
            </host:add>';
        
            $from[] = '/{{ remIp }}/';
            $to[] = '<host:rem>
            <host:addr ip="v4">'.$ipold.'</host:addr>
            </host:rem>';
        
            $from[] = '/{{ clTRID }}/';
            $clTRID = str_replace('.', '', round(microtime(1), 3));
            $to[] = htmlspecialchars('host-update-' . $clTRID);
            $xml = preg_replace($from, $to, '<epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
            <command>
            <update>
                <host:update
                    xmlns:host="urn:ietf:params:xml:ns:host-1.0">
                    <host:name>{{ name }}</host:name>
                    {{ addIp }}
                    {{ remIp }}
                </host:update>
            </update>
            <clTRID>{{ clTRID }}</clTRID>
            </command></epp>');
        
            $response=$this->runCommand($xml, __FUNCTION__,true);
            
            return $response;

        }
        
      
        
    }

    return true;
 
    // $cltRID=str_replace('.', '', round(microtime(1), 3));
    
    // $xml='<epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
    // <command>
    // <update>
    //   <host:update
    //    xmlns:host="urn:ietf:params:xml:ns:host-1.0">
    //     <host:name>'.$name.'</host:name>
    //     <add>
    //     <host:addr ip="v4">'.$ipv4.'</host:addr>
    //     </add>
    //   </host:update>
    // </update>
    // <clTRID>'.$cltRID.'</clTRID>
    // </command>
    // </epp>
    // ';
    
  
    
}

public function hostCheck($name){

$cltRID=str_replace('.', '', round(microtime(1), 3));
$xml='<epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
<command>
<check>
  <host:check
   xmlns:host="urn:ietf:params:xml:ns:host-1.0">
    <host:name>'.$name.'</host:name>
  </host:check>
</check>
<clTRID>'.$cltRID.'</clTRID>
</command>
</epp>
';

$response=$this->runCommand($xml, __FUNCTION__,true);
return  $response;

}


public function hostInfo($name){

    $cltRID=str_replace('.', '', round(microtime(1), 3));
    
    $xml='<epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
    <command>
    <info>
      <host:info
       xmlns:host="urn:ietf:params:xml:ns:host-1.0">
        <host:name>'.$name.'</host:name>
      </host:info>
    </info>
    <clTRID>'.$cltRID.'</clTRID>
    </command>
    </epp>
    ';
    
    $response=$this->runCommand($xml, __FUNCTION__,true);
    return  $response;
    
}



public function hostDelete($name){

    $cltRID=str_replace('.', '', round(microtime(1), 3));
    
    $xml='<epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
    <command>
    <delete>
      <host:delete
       xmlns:host="urn:ietf:params:xml:ns:host-1.0">
        <host:name>'.$name.'</host:name>
      </host:delete>
    </delete>
    <clTRID>'.$cltRID.'</clTRID>
    </command>
    </epp>
    ';
    
    $response=$this->runCommand($xml, __FUNCTION__,true);
    return  $response;
    
}


    public function create_hindidomain($domainName, $periodYr,$email,$password,$unique){

        $user = User::find($unique);

        $usercontact=DB::table('user_contact_details')->select('*')->where('user_id',$user->id)->first();
    
        $contactsXml="";
    
        $contents = [
            'registrant',
            'admin',
            'tech',
            'billing',
        ];
        
        foreach ($contents as $contactType) {
    
            $org=$user->institute;
            $name='';
            $address=$usercontact->address;
            $city=$usercontact->city;
            $state=$usercontact->state;
            $postcode=$usercontact->pincode;
            $email='';
            $mobileno='';
    
            if($contactType=='registrant'){
                $uniquecid=substr($user->domainPunycode,0,3).'R'.rand(1999045,9934777);
    
                $name=$user->name;
                $email=$user->email;
                $mobileno=$usercontact->mobile;
              
            }else if($contactType=='admin'){
                $uniquecid=substr($user->domainPunycode,0,3).'A'.rand(1999045,9934777);
    
                $value=json_decode($usercontact->administrative_detail,true);
    
                $name=$value['name']??'';
                $email=$value['email']??'';
                $mobileno=$value['mobile']??'';
    
            }else if($contactType=='tech'){
                $uniquecid=substr($user->domainPunycode,0,3).'T'.rand(1999045,9934777);
    
                $value=json_decode($usercontact->technical_detail,true);
    
                $name=$value['name']??'';
                $email=$value['email']??'';
                $mobileno=$value['mobile']??'';
    
            }else if($contactType=='billing'){
                $uniquecid=substr($user->domainPunycode,0,3).'B'.rand(1999045,9934777);
    
                $value=json_decode($usercontact->billing_contact_detail,true);
    
                $name=$value['name']??'';
                $email=$value['email']??'';
                $mobileno=$value['mobile']??'';
    
            }
    
            $modifiedText = str_replace("\n", "", $address);
    
            $params=[
                'name'=>$name,
                'org'=>$this->removeSpecialCharacters($org),
                'address'=>$this->removeSpecialCharacters($address),
                'city'=>$this->removeSpecialCharacters($city),
                'state'=>$this->removeSpecialCharacters($state),
                'postcode'=>$postcode,
                'country'=>'IN',
                'mobile'=>'+91.'.$mobileno,
                'email'=>$email,
                'password'=>$password
            ];
        
            
            $test = $this->contactTypeCreate($uniquecid,$params);            
    
    
            if ($contactType == 'registrant') {
                $contactsXml .= '<domain:registrant>' . $uniquecid . '</domain:registrant>';
            } else {
                $contactsXml .= '<domain:contact type="' . $contactType . '">' . $uniquecid . '</domain:contact>' . "\n";
            }
          
        }
       
        $authInfo=$password;
        $from = $to = [];

        $from[] = '/{{ name }}/';
        $to[] = htmlspecialchars($domainName);

        $from[] = '/{{ period }}/';
        $to[] = is_numeric($periodYr) ? $periodYr : 1;

        $from[] = '/{{ contacts }}/';
        $to[] = $contactsXml;
    
        $from[] = '/{{ authInfoPw }}/';
        $to[] = htmlspecialchars($authInfo);

        $from[] = '/{{ clTRID }}/';
        $clTRID = str_replace('.', '', round(microtime(1), 3));
        $to[] = htmlspecialchars("ernet" . '-domain-create-' . $clTRID);

        $from[] = "/<\w+:\w+>\s*<\/\w+:\w+>\s+/ims";
        $to[] = '';

        $xml = preg_replace($from, $to, '<epp xmlns="urn:ietf:params:xml:ns:epp-1.0"><command>
    <create>
      <domain:create
       xmlns:domain="urn:ietf:params:xml:ns:domain-1.0">
        <domain:name>{{ name }}</domain:name>
        <domain:period unit="y">{{ period }}</domain:period>
        {{ contacts }}
        <domain:authInfo>
          <domain:pw>{{ authInfoPw }}</domain:pw>
        </domain:authInfo>
      </domain:create>
    </create>
<extension>
 <idn:data
 xmlns:idn="urn:ietf:params:xml:ns:idn-1.0">
 <idn:table>hin-deva</idn:table>
 <idn:uname>{{ name }}</idn:uname>
 </idn:data>
 </extension>
    <clTRID>{{ clTRID }}</clTRID>
  </command>
  </epp>');

  

  $response=$this->runCommand($xml, __FUNCTION__,true);

  return  $response;

    }
    
    public function domainaddNameServer($domainName,$namservers){
    
        $cltRID=str_replace('.', '', round(microtime(1), 3));

        $hostxml = "";

        if (!empty($namservers)) {
            foreach($namservers as  $key=> $host){
                $res=$this->hostCreate($host['host'],$host['ip4'],$host['ip6']);
                $hostxml .= '<hostObj>' . $host['host'] . '</hostObj>' . "\n";
            }
        }
        
        $from[] = '/{{ hosts }}/';
        $to[] = $hostxml;

        $from[] = '/{{ clTRID }}/';
        $clTRID = str_replace('.', '', round(microtime(1), 3));
        $to[] = htmlspecialchars('er' . '-name-server-' . $clTRID);
    
        $xml=preg_replace($from, $to,'<epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
        <command>
        <update>
        <update xmlns="urn:ietf:params:xml:ns:domain-1.0">
        <name>'.$domainName.'</name>
        <add>
        <ns>
        '.$hostxml.'
        </ns>
        </add>
        </update>
        </update>
        <clTRID>{{clTRID}}</clTRID></command>
        </epp>');

        // Log::info($xml);

        $response=$this->runCommand($xml, __FUNCTION__,true);

        return  $response;
        
    }


    public function domainupdateNameServer($domainName,$namservers,$adddns,$remdns){
    
        $cltRID=str_replace('.', '', round(microtime(1), 3));

        $hostxml = "";

        if (!empty($namservers)) {
            foreach($namservers as  $key=> $host){
                $res=$this->hostCreate($host['host'],$host['ip4'],$host['ip6']);
            }
        }

        $hostaddxml='';
        if (!empty($adddns)) {
            foreach($adddns as $ns){
                $hostaddxml .= '<hostObj>' . $ns . '</hostObj>' . "\n";
            }
        }

        $hostremxml='';
        if (!empty($remdns)) {
            foreach($remdns as $ns){
                $hostremxml .= '<hostObj>' . $ns . '</hostObj>' . "\n";
            }
        }

        $from[] = '/{{ clTRID }}/';
        $clTRID = str_replace('.', '', round(microtime(1), 3));
        $to[] = htmlspecialchars('er' . '-name-server-' . $clTRID);
       
        $xml = preg_replace($from, $to, '<epp xmlns="urn:ietf:params:xml:ns:epp-1.0"><command>
        <update>
            <update xmlns="urn:ietf:params:xml:ns:domain-1.0">

                <name>'.$domainName.'</name>

                <rem><ns>
                '.$hostremxml.'
                </ns></rem>

                </update>
        </update>
        <clTRID>{{ clTRID }}</clTRID>
    </command> </epp>');
     
    $res=$this->runCommand($xml, __FUNCTION__,true);

    $xml2 = preg_replace($from, $to, '<epp xmlns="urn:ietf:params:xml:ns:epp-1.0"><command>
    <update>
        <update xmlns="urn:ietf:params:xml:ns:domain-1.0">
            <name>'.$domainName.'</name>

    <add><ns>
    '.$hostaddxml.'
    </ns></add>

            </update>
    </update>
    <clTRID>{{ clTRID }}</clTRID>
    </command> </epp>');
 

    $response=$this->runCommand($xml2, __FUNCTION__,true);

        // Log::info($xml);

        return  $response;

    }


    public function hostCreateLatest($name,$ipv4=null,$ipv6=null){

        $cltRID=str_replace('.', '', round(microtime(1), 3));
        if(($ipv6==':::::::' || $ipv6=':::::' || $ipv6==null ) && $ipv4!=null){
        
        $xml='<epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
        <command>
        <create>
          <host:create
           xmlns:host="urn:ietf:params:xml:ns:host-1.0">
            <host:name>'.$name.'</host:name>
            <host:addr ip="v4">'.$ipv4.'</host:addr>
          </host:create>
        </create>
        <clTRID>'.$cltRID.'</clTRID>
        </command>
        </epp>
        ';
        
        }else if($ipv4==null && ($ipv6==':::::' || $ipv6==null || $ipv6=':::::::'))
        {
            
        $xml='<epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
        <command>
        <create>
          <host:create
           xmlns:host="urn:ietf:params:xml:ns:host-1.0">
            <host:name>'.$name.'</host:name>
          </host:create>
        </create>
        <clTRID>'.$cltRID.'</clTRID>
        </command>
        </epp>
        ';
            
        }else{
        
        $xml='<epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
        <command>
        <create>
          <host:create
           xmlns:host="urn:ietf:params:xml:ns:host-1.0">
            <host:name>'.$name.'</host:name>
            <host:addr ip="v4">'.$ipv4.'</host:addr>
            <host:addr ip="v6">'.$ipv6.'</host:addr>
          </host:create>
        </create>
        <clTRID>'.$cltRID.'</clTRID>
        </command>
        </epp>
        ';
        
        }

        $response=$this->runCommand($xml, __FUNCTION__,true);
        
        $result=xmlConverter($response);

        $r=$result['epp']['response'];

        $code=$r['result_attr']['code'];

        if($code=='2302'){

           $res=$this->hostUpdate($name,$ipv4);

        }
       
        return $response;
        
    }

    public function domainupdateNameServerLatest($domainName,$namservers,$adddns,$remdns){
    
        $cltRID=str_replace('.', '', round(microtime(1), 3));

        $hostxml = "";

        if (!empty($namservers)) {
            foreach($namservers as  $key=> $host){
                $res=$this->hostCreateLatest($host['host'],$host['ip4'],$host['ip6']);
            }
        }

        $hostaddxml='';
        if (!empty($adddns)) {
            foreach($adddns as $ns){
                $hostaddxml .= '<hostObj>' . $ns . '</hostObj>' . "\n";
            }
        }

        $hostremxml='';
        if (!empty($remdns)) {
            foreach($remdns as $ns){
                $hostremxml .= '<hostObj>' . $ns . '</hostObj>' . "\n";
            }
        }

        $from[] = '/{{ clTRID }}/';
        $clTRID = str_replace('.', '', round(microtime(1), 3));
        $to[] = htmlspecialchars('er' . '-name-server-' . $clTRID);
       
        $xml = preg_replace($from, $to, '<epp xmlns="urn:ietf:params:xml:ns:epp-1.0"><command>
        <update>
            <update xmlns="urn:ietf:params:xml:ns:domain-1.0">

                <name>'.$domainName.'</name>

                <rem><ns>
                '.$hostremxml.'
                </ns></rem>

                </update>
        </update>
        <clTRID>{{ clTRID }}</clTRID>
    </command> </epp>');
     
    
    $res=$this->runCommand($xml, __FUNCTION__,true);

    $xml2 = preg_replace($from, $to, '<epp xmlns="urn:ietf:params:xml:ns:epp-1.0"><command>
    <update>
        <update xmlns="urn:ietf:params:xml:ns:domain-1.0">
            <name>'.$domainName.'</name>

    <add><ns>
    '.$hostaddxml.'
    </ns></add>

            </update>
    </update>
    <clTRID>{{ clTRID }}</clTRID>
    </command> </epp>');
 

    $response=$this->runCommand($xml2, __FUNCTION__,true);

        return  $response;

    }


    public function domainupdateDNSSEC($domainName,$adddnss){

        $cltRID=str_replace('.', '', round(microtime(1), 3));

        $addseq='';

        if (!empty($adddnss)) {
            foreach($adddnss as  $key=> $adddns){
                $addseq.="<dsData>
                <keyTag>" . $adddns['keyTag'] . "</keyTag>
                <alg>" . $adddns['alg'] . "</alg>
                <digestType>" . $adddns['digestType'] . "</digestType>
                <digest>" . $adddns['digest'] . "</digest>
                </dsData>". "\n";
            }
        }
    
        $from[] = '/{{ clTRID }}/';
        $clTRID = str_replace('.', '', round(microtime(1), 3));
        $to[] = htmlspecialchars('er' . '-name-server-' . $clTRID);
    
        $xml='<epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
        <command>
        <update>
        <update xmlns="urn:ietf:params:xml:ns:domain-1.0">
        <name>'.$domainName.'</name>
        </update>
        </update>
        <extension>
        <update xmlns="urn:ietf:params:xml:ns:secDNS-1.1">
      
        <add>
        '.$addseq.'
        </add>

        </update>
        </extension>
        <clTRID>ABC-12345</clTRID>
        </command>
        </epp>';

        // Log::info($xml);

        $response=$this->runCommand($xml, __FUNCTION__,true);

        return  $response;

    }


public function domainremDNSSEC($domainName,$remdnss){

    $cltRID=str_replace('.', '', round(microtime(1), 3));
    
    $remseq = "";

    if (!empty($remdnss)) {
        foreach($remdnss as  $key=> $remdnss){
            $remseq.="<dsData>
            <keyTag>" . $remdnss['keyTag'] . "</keyTag>
            <alg>" . $remdnss['alg'] . "</alg>
            <digestType>" . $remdnss['digestType'] . "</digestType>
            <digest>" . $remdnss['digest'] . "</digest>
            </dsData>". "\n";
        }
    }

    $from[] = '/{{ clTRID }}/';
    $clTRID = str_replace('.', '', round(microtime(1), 3));
    $to[] = htmlspecialchars('er' . '-name-server-' . $clTRID);

    $xml='<epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
    <command>
    <update>
    <update xmlns="urn:ietf:params:xml:ns:domain-1.0">
    <name>'.$domainName.'</name>
    </update>
    </update>
    <extension>
    <update xmlns="urn:ietf:params:xml:ns:secDNS-1.1">
  
    <rem>
    '.$remseq.'
    </rem>

    </update>
    </extension>
    <clTRID>ABC-12345</clTRID>
    </command>
    </epp>';

    // Log::info($xml);

    $response=$this->runCommand($xml, __FUNCTION__,true);

    return  $response;

}


public function domainStatusUpdate($domainName,$status,$mode){

    $cltRID=str_replace('.', '', round(microtime(1), 3));
    
    $xml='<epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
    <command>

    <update>
    <update xmlns="urn:ietf:params:xml:ns:domain-1.0">

        <name>'.$domainName.'</name>

        <'.$mode.'>
        <status s="'.$status.'" lang="en">'.$status.'</status>
        </'.$mode.'>

        </update>
    </update>
    <clTRID>'.$cltRID.'</clTRID>
    </command>
    </epp>
    ';
    
    $response=$this->runCommand($xml, __FUNCTION__,true);
    return  $response;

}


}

?>